<?php

/**
 * Handles Open Authorization functionality
 *
 * @author stuartlb3
 */
class NF_Helpscout_Authorization_Oauth
{

    /**
     * Dependency injector
     * @var array
     */
    protected $injector;

    /**
     * Constants object
     * @var NF_Helpscout_Admin_Constants
     */
    protected $constants;

    /**
     * Credentials object
     * @var NF_Helpscout_Authorization_Credentials
     */
    protected $credentials;

    /**
     * Order in which authorization will be attempted
     * @var array
     */
    protected $authorization_order = array();

    /**
     * URL that initiates an OAuth authorization code request
     * @var string
     */
    protected $initiate_oauth_url = '';

    /**
     * Body array for making Grant Type requests
     * @var array
     */
    protected $granttype_body = array();

    /**
     * Handles Open Authorization functionality
     * @param array $injector
     */
    public function __construct($injector)
    {
        $this->injector = $injector;

        $this->constants = $this->injector[ 'constants' ];

        $this->credentials = $this->injector[ 'credentials' ];

        $this->setAuthorizationParameters();
    }

    /**
     * Request access token using authorization code
     *
     * Internal method; called by public method
     *
     * @return array Response array of communication result
     */
    protected function grantTypeAuthorizationCode()
    {
        $this->setGrantTypeIdSecret();

        $this->granttype_body[ 'grant_type' ] = 'authorization_code';

        $this->granttype_body[ 'code' ] = $this->credentials->getAuthCode();

        $response = wp_remote_post($this->constants->token_url, array( 'body' => $this->granttype_body ));

        $processed_response_object = new NF_Helpscout_Authorization_OAuthResponseHandler($response, 'granttype_authcode');

        $response_array = $processed_response_object->getResponseArray();

        // check for token array
        // clear authorization code after authorizing
        if (isset($response_array[ 'api_data' ][ 'refresh_token' ])) {

            $this->credentials->clearAuthCode();

            $this->updateTokens($response_array[ 'api_data' ]);
        }

        return $response_array;
    }

    /**
     * Request access token using refresh token
     *
     * Internal method; called by public method
     *
     * @return array Response array of communication result
     */
    protected function grantTypeRefreshToken()
    {
        $this->setGrantTypeIdSecret();

        $this->granttype_body[ 'grant_type' ] = 'refresh_token';

        $this->granttype_body[ 'refresh_token' ] = $this->credentials->getRefreshToken();

        $response = wp_remote_post($this->constants->token_url, array( 'body' => $this->granttype_body ));

        $processed_response_object = new NF_Helpscout_Authorization_OAuthResponseHandler($response, 'granttype_refreshtoken');

        $response_array = $processed_response_object->getResponseArray();

        // check for token array
        if (isset($response_array[ 'api_data' ][ 'refresh_token' ])) {
            $this->updateTokens($response_array[ 'api_data' ]);
        }

        return $response_array;
    }

    /**
     * Updates the refresh and access tokens given a token array
     * @param array $token_array
     */
    protected function updateTokens($token_array)
    {
        $this->credentials->updateRefreshToken($token_array[ 'refresh_token' ]);

        if (isset($token_array[ 'access_token' ])) {
            $this->credentials->updateAccessToken($token_array[ 'access_token' ]);
        }
    }

    /**
     * Sets order in which to attempt communication
     */
    protected function setAuthorizationParameters()
    {
        if ($this->credentials->getClientId() && $this->credentials->getClientSecret()) {

            $this->setInitiateOauthURL();

            $this->setGrantTypeIdSecret();

            // reset to empty
            $this->authorization_order = array();

            if ($this->credentials->getAccessToken()) {
                $this->authorization_order[] = 'access_token';
            }

            if ($this->credentials->getRefreshToken()) {
                $this->authorization_order[] = 'granttype_refresh_token';
            }

            if ($this->credentials->getAuthCode()) {
                $this->authorization_order[] = 'granttype_authorization_code';
            }
        }

        if ($this->credentials->getAPIKey()) {
            $this->authorization_order[] = 'api_key';
        }
    }

    /**
     * Sets the client Id and secret for grant type requests
     *
     */
    protected function setGrantTypeIdSecret()
    {
        $this->granttype_body = array(
            'client_id' => $this->credentials->getClientId(),
            'client_secret' => $this->credentials->getClientSecret()
        );
    }

    /**
     * Sets the URL for initiating OAuth authorization code
     */
    protected function setInitiateOauthURL()
    {
        $this->initiate_oauth_url = $this->constants->initiate_oauth_url;

        $this->initiate_oauth_url .= '?client_id=' . $this->credentials->getClientId();
    }

    /**
     * Cycles through auth methods to generate access token
     * 
     * @return array|boolean Type of authorization, false if none available 
     */
    protected function cycleAuthorizationMethods()
    {
        // Declare default response outside of foreach to account for empty array.
        $return = false;

        foreach ($this->authorization_order as $auth_attempt) {

            if ('access_token' === $auth_attempt) {

                $return = array( 'access_token' => $this->credentials->getAccessToken() );
                // no need to continue
                break;
            }

            if ('granttype_refresh_token' === $auth_attempt) {

                $this->grantTypeRefreshToken();

                if ($this->credentials->getAccessToken()) {

                    $return = array( 'access_token' => $this->credentials->getAccessToken() );
                    break;
                }
            }

            if ('api_key' === $auth_attempt) {

                $return = array( 'api_key' => $this->credentials->getAPIKey() );
                break;
            }
        }

        return $return;
    }

    /**
     * Returns the OAuth URL for initiating authorization
     *
     * Empty string if client Id is not set
     * @return string
     */
    public function getInitiateOauthURL()
    {
        return $this->initiate_oauth_url;
    }

    /**
     * Request tokens using authorization code
     *
     * Includes full response array; use ['api_data'] for tokens
     * @return array
     */
    public function authorizeByAuthCode()
    {
        $response = $this->grantTypeAuthorizationCode();

        return $response;
    }

    /**
     * Request tokens using refresh token
     *
     * Includes full response array; use ['api_data'] for tokens
     * @return array Response array
     */
    public function authorizeByRefreshToken()
    {
        $response = $this->grantTypeRefreshToken();

        return $response;
    }

    /**
     * Returns array of authorization type and value, false if none available
     * 
     * Possible values:
     * 
     *  array['access_token'] => 'xxxxxx'
     * 
     *  array['api_key']='xxxxx'
     * 
     *  FALSE
     * 
     * @return array|boolean
     */
    public function getAuthorization()
    {
        $return = $this->cycleAuthorizationMethods();

        return $return;
    }

}
