<?php

/**
 * Extracts details from OAuth response
 *
 * @author stuartlb3
 */
class NF_Helpscout_Authorization_OAuthResponseHandler
{

    /**
     * Complete wp_remote_post response
     * @var array
     */
    protected $response;

    /**
     * Context of response
     * 
     * Describes the request made
     * 
     * @var string
     */
    protected $context;

    /**
     * Array of response body
     * 
     * @var array 
     */
    protected $body;

    /**
     * Array of all response information
     * @var array
     */
    protected $response_array;

    /**
     * Stops the processing of evaluateResponse
     * 
     * @var boolean
     */
    protected $continue_evaluate_response;

    /**
     * Extracts details from OAuth response
     * @param array|WP_Error $response
     * @param string $context
     */
    public function __construct($response, $context)
    {
        $this->response = $response;

        $this->context = $context;

        $this->initializeProperties();

        switch ($context) {
            case 'granttype_authcode':
            case 'granttype_refreshtoken':
                $this->processTokenResponse();
                break;
        }
    }

    /**
     * Checks for WP_error, then extracts token or error messages
     */
    protected function processTokenResponse()
    {
        $this->wpErrorCheck();

        if ($this->continue_evaluate_response) {

            $this->body = json_decode($this->response[ 'body' ], TRUE);

            $this->response_array[ 'response_body' ] = $this->body;

            $this->checkForTokens();
        }

        // Tokens not present
        if ($this->continue_evaluate_response) {
            $this->checkForRejections();
        }
    }

    /**
     * Check for refresh token, add to response
     * 
     * Stop evaluation if present as no need to continue
     */
    protected function checkForTokens()
    {
        if (isset($this->body[ 'refresh_token' ])) {

            $this->response_array[ 'api_data' ] = $this->body;

            $this->response_array[ 'status' ] = 'success';

            $this->continue_evaluate_response = false;
        }
    }

    /**
     * Extract error responses if error codes are in known locations
     */
    protected function checkForRejections()
    {
        if (isset($this->body[ 'error' ])) {

            $this->setErrorResponses('api_error');

            $this->response_array[ 'error_codes' ][] = $this->body[ 'error' ];

            if (isset($this->body[ 'error_description' ])) {
                $this->response_array[ 'error_messages' ][] = $this->body[ 'error_description' ];
            }

            $this->continue_evaluate_response = FALSE;
        }
    }

    /**
     * Check for WP_Error; add error messages to response array
     */
    protected function wpErrorCheck()
    {
        if (is_wp_error($this->response)) {

            $this->setErrorResponses('wp_error');

            $error_mesages = $this->response->get_error_messages();

            if (!empty($error_mesages)) {

                $this->response_array[ 'error_messages' ] = $error_mesages;
            }

            $this->response_array[ 'status' ] = __('WordPress error', 'ninja-forms-zoho-crm');
        }
    }

    /**
     * Set standard error values in response array
     * @param string $error_type Specific error type - wp_error or api_error
     */
    protected function setErrorResponses($error_type)
    {
        $this->response_array['status']= 'failure';
        
        $this->response_array[ 'error' ] = TRUE;

        $this->response_array[ $error_type ] = TRUE;

        $this->continue_evaluate_response = FALSE;
    }

    /**
     * The method name is literally the description
     */
    protected function initializeProperties()
    {
        $this->continue_evaluate_response = TRUE;

        $this->response_array = array(
            'api_data' => array(), // Response data
            'timestamp' => '',
            /**
             * String description of result
             */
            'status' => '',
            'error' => false, // Any type of error
            'wp_error' => false, // Wordpress error object
            'api_error' => false, // Comm error using API
            /**
             * Indexed array of error codes string elements
             * returned from API or internally generated
             */
            'error_codes' => array(),
            /**
             * Indexed array of Error messages string elements
             * Returned from API or internally generated
             */
            'error_messages' => array(),
            'headers' => array(),
            'response_body' => '' // Full Response
        );

        $this->response_array[ 'timestamp' ] = date(DATE_W3C);
    }

    /**
     * Returns response array of communication result
     * @return array
     */
    public function getResponseArray()
    {
        return $this->response_array;
    }

}
