<?php

/**
 * Provides the authorization header required for making API request
 *
 * @author stuartlb3
 */
class NF_Helpscout_Authorization_Header
{

    /**
     * Dependency injector
     * @var array
     */
    protected $injector;

    /**
     * Constants object
     * @var NF_Helpscout_Admin_Constants
     */
    protected $constants;

    /**
     * Credentials used for authorization
     * 
     * @var NF_Helpscout_Authorization_Credentials
     */
    protected $credentials;

    /**
     * OAuth object
     * @var NF_Helpscout_Authorization_Oauth
     */
    protected $oauth;

    /**
     * Endpoint URL for requests
     * 
     * Different endpoint for V1 vs V2
     * @var string
     */
    protected $endpoint_url = '';

    /**
     * Endpoint URL for mailbox list
     * 
     * Different endpoint for V1 vs V2
     * @var string
     */
    protected $mailbox_list_url = '';

    /**
     * Header authorization structured as array for wp_remote_post 'headers'
     * 
     * Structure is:
     *  array( 'Authorization' =>{authorization goes here} )
     * 
     * Optionally can add other headers, e.g. content-type
     * @var array 
     */
    protected $authorization_header;

    public function __construct($injector)
    {
        $this->injector = $injector;

        $this->constants = $this->injector[ 'constants' ];

        $this->credentials = $this->injector[ 'credentials' ];

        $this->oauth = $this->injector[ 'oauth' ];

        $this->authorization_header = array( 'Authorization' => '' );
    }

    /**
     * Sets the header and endpoints based on authorization
     * 
     * If OAuth object has an access token, then use API V2 with OAuth
     */
    protected function setHeaderAndEndpoints()
    {
        $auth = $this->oauth->getAuthorization();

        if (isset($auth[ 'access_token' ])) {

            $this->setAuthorizationAccesstoken($auth[ 'access_token' ]);

            $this->setURLHelpscoutV2();
            return;
        } elseif (isset($auth[ 'api_key' ])) {

            $this->setAuthorizationHelpscoutV1($auth[ 'api_key' ]);

            $this->setURLHelpscoutV1();
            return;
        }
        
        // no authorization is set, default to V2
        // empty access token will fail, but not throw WP error            
        // Issue #31
            $this->setAuthorizationAccesstoken('');
            $this->setURLHelpscoutV2();
        
    }

    /**
     * Sets authorization header using access token
     * 
     * @param string $access_token
     */
    protected function setAuthorizationAccesstoken($access_token)
    {
        $this->authorization_header[ 'Authorization' ] = 'Bearer ' . $access_token;
    }

    /**
     * Sets URL endpoint for V2 communication
     */
    protected function setURLHelpscoutV2()
    {
        $this->endpoint_url = $this->constants->v2_conversations_endpoint;
        $this->mailbox_list_url = $this->constants->v2_mailboxes_endpoint;
    }

    /**
     * Sets authorization header using V1 API key credentials
     */
    protected function setAuthorizationHelpscoutV1($api_key)
    {
        $api_key = $this->credentials->getAPIKey();

        $this->authorization_header[ 'Authorization' ] = 'Basic ' . base64_encode($api_key . ':X');
    }

    /**
     * Sets URL endpoint for V2 communication
     */
    protected function setURLHelpscoutV1()
    {
        $this->endpoint_url = $this->constants->v1_conversations_endpoint;
        $this->mailbox_list_url = $this->constants->v1_mailboxes_endpoint;
    }

    /**
     * Sets content-type to JSON
     * 
     * Default content type
     */
    protected function setContentTypeJSON()
    {
        $this->authorization_header[ 'content-type' ] = 'application/json';
    }

    /**
     * Returns the authorization header
     * 
     * Sets header only on command because auth object will attempt to generate
     * new access token if old one is expired so don't make API call unless
     * requested. 
     */
    public function getAuthHeader()
    {
        $this->setHeaderAndEndpoints();

        $this->setContentTypeJSON();

        return $this->authorization_header;
    }

    /**
     * Returns endpoint as set based on authorization level
     * 
     * Uses V2 if using OAuth, V1 if using API key
     * @return string
     */
    public function getEndpointURL()
    {
        if (0 == strlen($this->endpoint_url)) {
            $this->setHeaderAndEndpoints();
        }

        return $this->endpoint_url;
    }
    /**
     * Returns endpoint as set based on authorization level
     * 
     * Uses V2 if using OAuth, V1 if using API key
     * @return string
     */
    public function getMailboxListURL()
    {
        if (0 == strlen($this->mailbox_list_url)) {
            $this->setHeaderAndEndpoints();
        }

        return $this->mailbox_list_url;
    }
}
