<?php

/**
 * Retrieves and stores credentials
 *
 * @author stuartlb3
 */
class NF_Helpscout_Authorization_Credentials
{

    /**
     * Constants
     * @var NF_Helpscout_Admin_Constants
     */
    protected $constants;

    /**
     * API Key
     * @var string|mixed
     */
    protected $apikey = null;

    /**
     * Access token
     * @var string|mixed
     */
    protected $access_token = null;

    /**
     * Refresh token
     * @var string|mixed
     */
    protected $refresh_token = null;

    /**
     * Client Id
     * @var string|mixed
     */
    protected $client_id = null;

    /**
     * Client secret
     * @var string|mixed
     */
    protected $client_secret = null;

    /**
     * Authorization code
     * @var string|mixed
     */
    protected $authcode = null;

    /**
     * Retrieves and stores credentials
     * 
     * Values are not retrieved on construction to reduce db calls.  Each 
     * property starts of null and the first public request for it triggers
     * a method call to retrieve the value, at which time it is stored as 
     * a property and and then be retrieved w/o making additional db requests
     * 
     * @param array $injector
     */
    public function __construct($injector)
    {
        $this->constants = $injector[ 'constants' ];
    }

    /**
     * Retrieves API Key from NF Settings
     */
    protected function retrieveAPIKey()
    {
        $key = $this->constants->apikey_key;

        $this->apikey = Ninja_Forms()->get_setting($key, false);

        if (0 == strlen($this->apikey)) {
            $this->apikey = FALSE;
        }
    }

    /**
     * Retrieves access token from WP options table transient
     *
     * If the transient does not exist, does not have a value, or has expired,
     * then get_transient will return false
     */
    protected function retrieveAccessToken()
    {
        $key = $this->constants->access_token_key;

        $this->access_token = get_transient($key);

        // if transient is set but empty, set to false
        if (0 == strlen($this->access_token)) {
            $this->access_token = FALSE;
        }
    }

    /**
     * Update access token with new value
     * 
     * Set as transient to expire five minutes before token expiry
     * @param string $value
     */
    public function updateAccessToken($value)
    {
        $this->access_token = $value;
        
        $key = $this->constants->access_token_key;
        $expiry = $this->constants->access_expiry - (5*60);

        set_transient($key, $value, $expiry);
    }

    /**
     * Retrieves refresh token from WP options table
     */
    protected function retrieveRefreshToken()
    {
        $key = $this->constants->refresh_token_key;

        $this->refresh_token = get_option($key, false);

        if (0 == strlen($this->refresh_token)) {
            $this->refresh_token = FALSE;
        }
    }

    /**
     * Update refresh token
     * 
     * Stored in WP options table
     * @param string $value
     */
    public function updateRefreshToken($value)
    {
        $this->refresh_token = $value;
        
        $key = $this->constants->refresh_token_key;

        update_option($key, $value);
    }

    /**
     * Retrieve client Id from NF Settings
     */
    protected function retrieveClientId()
    {
        $key = $this->constants->client_id_key;

        $this->client_id = Ninja_Forms()->get_setting($key, false);

        if (0 == strlen($this->client_id)) {
            $this->client_id = FALSE;
        }
    }

    /**
     * Update client Id
     * 
     * Stored in NF Settings
     * @param string $value
     */
    public function updateClientId($value)
    {
        $this->client_id = $value;
        
        $key = $this->constants->client_id_key;

        Ninja_Forms()->update_setting($key, $value);
    }

    /**
     * Retrieves client secret from NF Settings
     */
    protected function retrieveClientSecret()
    {
        $key = $this->constants->client_secret_key;

        $this->client_secret = Ninja_Forms()->get_setting($key, false);

        if (0 == strlen($this->client_secret)) {
            $this->client_secret = FALSE;
        }
    }

    /**
     * Update client secret
     * 
     * Stored in NF Settings
     * @param string $value
     */
    public function updateClientSecret($value)
    {
        $this->client_secret =  $value;
        
        $key = $this->constants->client_secret_key;

        Ninja_Forms()->update_setting($key, $value);
    }

    /**
     * Retrieves authorization code from NF Settings
     */
    protected function retrieveAuthCode()
    {
        $key = $this->constants->authcode_key;

        $this->authcode = Ninja_Forms()->get_setting($key, false);

        if (0 == strlen($this->authcode)) {
            $this->authcode = FALSE;
        }
    }

    /**
     * Update authcode
     * 
     * Stored in NF Settings
     * @param string $value
     */
    public function updateAuthCode($value)
    {
        $this->authcode = $value;
        
        $key = $this->constants->authcode_key;
        
        Ninja_Forms()->update_setting($key, $value);
    }

    /**
     * Clears authcode
     * 
     * Authcode is extremely short lived and used only once.  Clear after use
     * to reduce confusion in OAuth authorization process
     */
    public function clearAuthCode()
    {
        $key = $this->constants->authcode_key;

        Ninja_Forms()->update_setting($key, '');
    }

    /**
     * Returns the stored API key, false if not set
     * @return string|false
     */
    public function getAPIKey()
    {
        if (is_null($this->apikey)) {
            $this->retrieveAPIKey();
        }

        return $this->apikey;
    }

    /**
     * Returns stored access token, false on fail
     * @return string|false
     */
    public function getAccessToken()
    {
        if (is_null($this->access_token)) {
            $this->retrieveAccessToken();
        }

        return $this->access_token;
    }

    /**
     * Returns stored refresh token, false if not set
     * @return string|false
     */
    public function getRefreshToken()
    {
        if (is_null($this->refresh_token)) {
            $this->retrieveRefreshToken();
        }

        return $this->refresh_token;
    }

    /**
     * Returns the stored client Id, false if not set
     * @return string|false
     */
    public function getClientId()
    {
        if (is_null($this->client_id)) {
            $this->retrieveClientId();
        }

        return $this->client_id;
    }

    /**
     * Returns the stored client secret, false if not set
     * @return string|false
     */
    public function getClientSecret()
    {
        if (is_null($this->client_secret)) {
            $this->retrieveClientSecret();
        }

        return $this->client_secret;
    }

    /**
     * Returns the stored authorization code, false if not set
     * @return string|false
     */
    public function getAuthCode()
    {
        if (is_null($this->authcode)) {
            $this->retrieveAuthCode();
        }

        return $this->authcode;
    }

}
