<?php

/**
 * Handles AJAX on OAuth settings page
 *
 * Checks POST data looking for a 'context' key value pair.  The value is used
 * to select the desired method for handling a given event.  The method sends
 * back an associative array of response values.
 *
 * @author stuartlb3
 */
class NF_Helpscout_Authorization_AJAX
{

    /**
     * Dependency injector
     * @var array
     */
    protected $injector;

    /**
     * Constants object
     * @var NF_Helpscout_Admin_Constants
     */
    protected $constants;

    /**
     * Credentials object
     * @var NF_Helpscout_Authorization_Credentials
     */
    protected $credentials;

    /**
     * OAuth object
     * @var NF_Helpscout_Authorization_Oauth
     */
    protected $oauth;

    /**
     * Handles AJAX on OAuth settings page
     *
     * @var array Dependency injector
     */
    public function __construct($injector)
    {
        $this->injector = $injector;

        $this->constants = $this->injector[ 'constants' ];

        $this->credentials = $this->injector[ 'credentials' ];

        $this->oauth = $this->injector[ 'oauth' ];
    }

    /**
     * Listens for POST data and returns a handled response
     *
     * @return array
     */
    public function respondToAJAX()
    {
        $context = filter_input(INPUT_POST, 'context', FILTER_SANITIZE_SPECIAL_CHARS);

        switch ($context) {
            case 'client_id_change':
                $response = $this->respondToClientIdChange();
                break;

            case 'client_secret_change':
                $response = $this->respondToClientSecretChange();
                break;

            case 'authcode_change':
                $response = $this->respondToAuthcodeChange();
                break;

            case 'clickedonauthcodegrant':
                $response = $this->respondToAuthCodeGrant();
                break;

            default:
                break;
        }

        $response[ 'context' ] = "The context is $context";

        return $response;
    }

    /**
     * Updates the NF Setting for client Id
     *
     * When the textbox is changed, update the stored value and the
     * credential object's property
     *
     * @return array
     */
    protected function respondToClientIdChange()
    {
        $value = filter_input(INPUT_POST, 'client_id', FILTER_SANITIZE_SPECIAL_CHARS);

        $this->credentials->updateClientId($value);

        return array();
    }

    /**
     * Updates the NF Setting for client Id
     *
     * When the textbox is changed, update the stored value and the
     * credential object's property
     *
     * @return array
     */
    protected function respondToClientSecretChange()
    {
        $value = filter_input(INPUT_POST, 'client_secret', FILTER_SANITIZE_SPECIAL_CHARS);

        $this->credentials->updateClientSecret($value);
    }

    /**
     * Updates the NF Setting for authorization code
     *
     * When the textbox is changed, update the stored value and the
     * credential object's property
     *
     * @return array
     */
    protected function respondToAuthcodeChange()
    {
        $value = filter_input(INPUT_POST, 'authcode', FILTER_SANITIZE_SPECIAL_CHARS);

        $this->credentials->updateAuthCode($value);
    }

    /**
     * Sends an OAuth "grant type = auth code" request
     *
     * @return array Updated access and refresh tokens
     */
    protected function respondToAuthCodeGrant()
    {
        $this->oauth->authorizeByAuthCode();

        $this->credentials->updateAuthCode('');

        $return = array(
            'access_token' => $this->credentials->getAccessToken(),
            'refresh_token' => $this->credentials->getRefreshToken(),
            'authcode' => $this->credentials->getAuthCode()
        );

        return $return;
    }

}
