<?php

/**
 * Modifies configured settings at time of settings display
 *
 * @author stuartlb3
 */
class NF_Helpscout_Admin_DynamicSettings
{

    /**
     * Dependency injector
     * @var array
     */
    protected $injector;

    /**
     * Credentials used for authorization
     *
     * @var NF_Helpscout_Authorization_Credentials
     */
    protected $credentials;

    /**
     * Handles Open Authorization functionality
     *
     * @var NF_Helpscout_Authorization_Oauth
     */
    protected $oauth;

    /**
     * Settings array during modification process
     * @var array
     */
    protected $dynamic_settings;

    /**
     * Modifies configured settings at time of settings display
     * @param array $injector Dependency injector
     */
    public function __construct($injector)
    {
        $this->injector = $injector;

        $this->credentials = $this->injector[ 'credentials' ];

        $this->oauth = $this->injector[ 'oauth' ];

        add_filter('nf_helpscout_plugin_settings', array( $this, 'filter_settings' ));
    }

    /**
     * Filter to receive, modify, and return configured plugin settings
     *
     * @param array $settings
     * @return array
     */
    public function filter_settings($settings)
    {
        $this->dynamic_settings = $settings;

        $this->markupButtonInitiateAuthorization();

        $this->markupButtonAuthcodeGrantURL();

        $this->markupRefreshToken();

        $this->markupAccessToken();

        return $this->dynamic_settings;
    }

    /**
     * Builds markup for button that initiates authorization process
     */
    protected function markupButtonInitiateAuthorization()
    {
        $link = $this->oauth->getInitiateOauthURL();

        if (0 !== strlen($link)) {

            $text = __('Click to initiate authorization', 'ninja-forms-helpscout');

            $markup = "<span id='helpscout_initiate_oauth'><a href='$link' >$text</a></span>";

            $this->dynamic_settings[ 'helpscout_initiate_authorization' ][ 'html' ] = $markup;
        }
    }

    /**
     * Builds markup for auth code grant URL
     */
    protected function markupButtonAuthcodeGrantURL()
    {
        $text = __('Generate access using authorization code', 'ninja-forms-helpscout');

        $this->dynamic_settings[ 'helpscout_authcode_grant_url' ][ 'html' ] = "<input type='button' id='helpscout_authcodegrant' value = '$text' />";
    }

    /**
     * Adds markup for refresh token
     * 
     * Sets html and desc on page load; these values can be modified with
     * AJAX should results change by page interaction
     */
    protected function markupRefreshToken()
    {
        if ($this->credentials->getRefreshToken()) {

            $markup = '<span id="helpscout_refresh_token" style="background-color:LightGreen">' . $this->credentials->getRefreshToken() . '</span>';

            $description = '<span id="helpscout_refresh_token_description">'
                    . __('You have a refresh token and are connected to Helpscout - good job!', 'ninja-forms-helpscout')
                    . '</span>';
        } else {
            $markup = '<span id="helpscout_refresh_token" style="background-color:LightYellow">' . 'Refresh token not set' . '</span>';

            $description = '<span id="helpscout_refresh_token_description">'
                    . __('Please complete the authorization process to generate a refresh token.  This token will enable communication to your Helpscout account.', 'ninja-forms-helpscout')
                    . '</span>';
        }

        $this->dynamic_settings[ 'helpscout_refresh_token' ][ 'html' ] = $markup;
        $this->dynamic_settings[ 'helpscout_refresh_token' ][ 'desc' ] = $description;
    }

    /**
     * Adds markup for access token
     * 
     * Sets html and desc on page load; these values can be modified with
     * AJAX should results change by page interaction
     */
    protected function markupAccessToken()
    {
        if ($this->credentials->getAccessToken()) {

            $markup = '<span id="helpscout_access_token">' . $this->credentials->getAccessToken() . '</span>';
        } else {
            $markup = '<span id="helpscout_access_token">' . 'Access token not set or expired' . '</span>';
        }
        $this->dynamic_settings[ 'helpscout_access_token' ][ 'html' ] = $markup;
    }

}
