<?php

/**
 * Analyze the Pipeline response and return all pertinent information
 *
 * 
 */
class PipelineHandleResponse {

    /**
     * The full response from Pipeline, regardless of outcome
     * 
     * Can include WP errors, Pipeline rejection errors, success messages,
     * IDs for newly created sections, data requests
     * @var mixed Full response from Pipeline 
     */
    protected $full_response;

    /**
     * Module name of the response to process
     * @var string
     */
    protected $module;

    /**
     * Body of the response in JSON for data extraction
     * @var Object 
     */
    protected $body_json;

    /**
     * The GET data contained in the body, extracted for additional processing
     * @var object
     */
    protected $body_json_data;

    /**
     * Array of boolean flags to describe the status
     * @var array 
     */
    protected $flag;

    /**
     * Array of readable summaries of communication result
     * @var array 
     */
    protected $response_summary = array();

    /**
     * The response nested array
     * @var integer 
     */
    protected $code;

    /**
     * The response message
     * @var string 
     */
    protected $message;

    /**
     * array of IDs of the newly created entities within a module
     * @var array 
     */
    protected $new_id = array();

    /**
     * Response from Pipeline
     * @param array|object $full_response 
     */
    public function __construct($module, $full_response) {

        $this->initialize_variables();

        $this->module = $module;

        $this->full_response = $full_response;

        if (is_wp_error($this->full_response)) {

            $this->handle_wp_error();
            return;
        }
        $this->body_json = json_decode($this->full_response['body']);

        $this->extract_code_message();

        $this->handle_codes();
    }

    protected function handle_codes() {

        switch ($this->code) {

            case 200: // OK
                $this->extract_200_data();
                break;

            case 401: // Unauthorized
                $this->build_401_message();
                break;

            case 422: // Unprocessable entity
                $this->build_422_message();
                break;

            default:
                $this->build_unhandled_responses();
                break;
        }
    }

    /**
     * Extract the response from a successful section entry
     */
    protected function extract_200_data() {

        if (isset($this->body_json->created_at)) {
            
            $created_at = $this->body_json->created_at;

            $this->response_summary[] = __('Successfully created at: ', 'ninja-forms-pipeline-crm') . $created_at;
            
            $this->build_new_id_array(); 
        }

        switch($this->module){
            case 'person_custom':           
            case 'company_custom': 
            case 'deal_custom':    

                $this->extract_account_data();
                break;
            
            default:
                break;
        }
    }

    protected function extract_account_data(){
        
        foreach($this->body_json->entries as $entry){
            $array = array(
                'id'=>$entry->id,
                'label'=>$entry->name,
                'field_type'=>$entry->field_type,    
            );
            
            if(isset($entry->custom_field_label_dropdown_entries)){
                
                $options = array();
                
                foreach($entry->custom_field_label_dropdown_entries as $option){
                    
                    $options[]=array(
                        'option_id'=>$option->id,
                        'option_label'=>$option->name,              
                    );
                }
                $array['options']=$options;
            }
            $this->body_json_data[]=$array;
        }      
    }
    
    /**
     * Build 422-Unprocessable
     */
    protected function build_422_message() {



        if ($this->body_json->error) {

            $this->message = $this->body_json->error;

            $this->response_summary[] = $this->message;
        } else {
            foreach ($this->body_json as $obj) {

                $this->response_summary[] = $obj->humanized_field . ' - ' . $obj->msg;

                $this->append_contextual_help($obj->msg, $obj->humanized_field);
            }
        }
    }

    /**
     * Build 401-Unauthorized Messages
     */
    protected function build_401_message() {

        $message = __('Please check your API Key.  It may be incorrect.', 'ninja-forms-pipeline-crm');
        $message .= '<br />' . $this->code . ' ' . $this->message;

        $this->response_summary[] = $message;
    }

    /**
     * Build response for unhandled response codes
     */
    protected function build_unhandled_responses() {

        $this->response_summary[] = $this->code . ' ' . $this->message;
    }

    /**
     * Extract the code and message from the response's "RESPONSE" 
     */
    protected function extract_code_message() {

        $this->code = $this->full_response['response']['code'];
        $this->message = $this->full_response['response']['message'];
    }

    protected function initialize_variables() {

        $this->flag = array(// initialize
            'wp_error' => false,
        );
    }

    protected function handle_wp_error() {

        $this->flag['wp_error'] = true;

        $code = $this->full_response->get_error_code();

        $message = $this->full_response->get_error_message();

        $this->response_summary[] = $code . ' - ' . $message;
    }

    /**
     * Builds the new_id array of module entries and their ids
     */
    protected function build_new_id_array() {


        if (isset($this->body_json->id)) {

            $this->new_id[(string) $this->module] = (integer) $this->body_json->id;
//                update_option('thisbodyjson',$this->body_json->id);
        }
    }

    /**
     * Appends contextual help to the response summary
     * 
     * @param string $string
     * @param string $detail
     */
    protected function append_contextual_help($string, $detail = '') {

        $support_object = NF_PipelineCRM_Constants::SUPPORT_CLASS;

        $contextual_help = $support_object->help($string, $detail);

        if ($contextual_help) {

            $this->response_summary[] = $contextual_help;
        }
    }

    /**
     * Returns response summary
     * 
     * @return boolean|array
     */
    public function get_response_summary() {

        if (!isset($this->response_summary)) {

            return false;
        } else {

            return $this->response_summary;
        }
    }

    /**
     * Returns array of new id integers, keyed on module
     * @return array 
     */
    public function get_new_id() {

        if (!isset($this->new_id)) {

            return false;
        } else {

            return $this->new_id;
        }
    }

    /**
     * Returns the JSON data
     * 
     * @return boolean|object
     */
    public function get_body_json_data() {

        if (!isset($this->body_json_data)) {

            return false;
        } else {

            return $this->body_json_data;
        }
    }

}
