<?php

class PipelineCreate {

    /**
     * Section into which new entity is created
     * @var string 
     */
    protected $module;

    /**
     * Incoming JSON request
     * @var string 
     */
    protected $json_request;

    /**
     * Keyed array of already-created modules and their IDs for linking
     * @var array
     */
    protected $new_id_array;

    /**
     * Configuration array keyed on module
     * @var array
     */
    protected $module_config;
    
    /**
     * Endpoint for request
     * @var string
     */
    protected $endpoint;
    
    /**
     * API Key for CRM Account
     * @var string 
     */
    protected $apikey;

    /**
     * Timestamp of last authorization communication
     * @var integer 
     */
    protected $timestamp;


    /**
     * Creates a new module in Pipeline
     * 
     * 
     * @param string $module Module for which the create request is made
     * @param string $json_request Incoming JSON request string
     */
    public function __construct($module, $json_request, $new_id_array = array()) {

        $this->module = $module;

        $this->json_request = $json_request;

        $this->new_id_array = $new_id_array;
        
        $this->module_config = NF_PipelineCRM()->get_module_config();
        
        $this->extract_credentials();
        
        $this->build_endpoint();

        $this->make_request();
    }

    /**
     * Retrieves login credentials
     */
    protected function extract_credentials() {

        $credentials = NF_PipelineCRM()->get_credentials();

        if (!$credentials['valid']) {

            return; // add some support message
        } else {

            $this->apikey = $credentials['apikey'];
        }
    }

    /**
     * Makes request to create entry
     */
    protected function make_request() {

        $url = NF_PipelineCRM_Constants::BASE_URL . $this->endpoint.'?api_key='.$this->apikey;

        $headers_array = array(
            'Content-Type' => 'application/json',
        );

        $args = array(
            'timeout' => 45,
            'redirection' => 0,
            'httpversion' => '1.0',
            'sslverify' => FALSE,
            'method' => 'POST',
            'headers' => $headers_array,
            'body' => $this->json_request
        );

        $this->full_response = wp_remote_post(
                $url, $args
        );
    }


    /**
     * Build the endpoint for a module request
     */
    protected function build_endpoint() {

        switch($this->module){

            default:
                $this->endpoint = $this->module_config[$this->module]['endpoint'];
        }
    }

    /**
     * Returns the full response array
     * @return mixed
     */
    public function get_full_response() {

        if (empty($this->full_response)) {
            return false;
        } else {

            return $this->full_response;
        }
    }

}
