<?php

if (!defined('ABSPATH'))
    exit;

if (!class_exists('NF_Abstracts_PaymentGateway'))
    return;

class NF_Elavon_PaymentGateway extends NF_Abstracts_PaymentGateway {

    protected $_slug = 'elavon';

    /**
     * Credit card data from submission handled separately for security
     * @var array
     */
    protected $credit_card_data;

    /**
     * Form processing object called on action process
     * @var object
     */
    protected $form_process;
    
    public function __construct() {

        parent::__construct();

        $this->_name = __('Elavon', 'ninja-forms-elavon-payment-gateway');

        $this->build_admin_settings();

        add_action('ninja_forms_builder_templates', array($this, 'builder_templates'));

        add_filter('ninja_forms_run_action_settings', array($this, 'iterate_receipt_placeholders'), 10, 4);

        add_action('ninja_forms_enqueue_scripts', array($this, 'enqueue_scripts'));
    }

    /**
     * Function to output display templates.
     */
    public function builder_templates() {

        NF_Elavon::template('custom-field-map-row.html');
    }

    /**
     * Build admin settings that need to be constructed instead of configured
     * 
     * Drop down selector for field map is constructed because the full array
     * for the field map includes both readable names and mapping instructions
     * and thus can't be simply configured with a formatted array
     */
    public function build_admin_settings() {

        // configure array of action settings
        $settings = NF_Elavon::config('ActionFieldMapSettings');

        $this->_settings = array_merge($this->_settings, $settings);


        $this->_settings[NF_Elavon_Constants::FIELD_MAP_REPEATER_KEY]['columns']['field_map']['options'] = $this->return_field_map_dropdown();
    }

    /**
     * Returns the field map dropdown array for the option repeater dropdown
     * 
     * @return array Field Map dropdown array for option repeater
     */
    protected function return_field_map_dropdown() {

        $field_map_array = NF_Elavon()->get_field_map_array();

        $field_map_dropdown = NF_Elavon_Functions::build_field_map_dropdown($field_map_array);

        return $field_map_dropdown;
    }

    /**
     * Process
     *
     * The main function for processing submission data.
     *
     * @param array $action_settings Action specific settings.
     * @param int $form_id The ID of the submitted form.
     * @param array $data Form submission data.
     * @return array $data Modified submission data.
     */
    public function process($action_settings, $form_id, $data) {

        $this->load_process_classes();

        $this->form_process = new ElavonProcessForm($action_settings, $form_id, $data);

        $post_action_data = $this->form_process->data();

        return $post_action_data;
    }

    /**
     * Loads the class files used in processing the form
     */
    protected function load_process_classes() {

        $array = array('ProcessForm', 'RequestPayment', 'HandleResponse', 'BuildReceipt');

        foreach ($array as $classfilename) {

            include_once NFELAVON_PLUGIN_DIR . 'includes/Comm/' . $classfilename . '.php';
        }
    }

    /**
     * 
     * @param array $action_settings
     * @param string $form_id
     * @param string $action_id
     * @param array $form_settings
     * @return array
     */
    public function iterate_receipt_placeholders($action_settings, $form_id, $action_id, $form_settings) {

        if (is_null($this->form_process) || '1' != $action_settings['active']) {

            return $action_settings;
        }

        $action_settings['success_msg'] = $this->replace_placehoders( $action_settings['success_msg']);

        $action_settings['email_message'] = $this->replace_placehoders( $action_settings['email_message']);

        $action_settings['email_message_plain'] =  $this->replace_placehoders( $action_settings['email_message_plain']);

        return $action_settings;
    }

    /**
     * Replaces merge tags with process-returned values
     * @param string $incoming
     * @return string
     */
    protected function replace_placehoders($incoming) {

        $wip = str_replace(NF_Elavon_Constants::HTML_RECEIPT_FIELD_KEY, $this->form_process->html_receipt(), $incoming);

        $wip = str_replace(NF_Elavon_Constants::PLAINTEXT_RECEIPT_FIELD_KEY, $this->form_process->plaintext_receipt(), $wip);

        $outgoing = str_replace(NF_Elavon_Constants::TXN_ID_FIELD_KEY, $this->form_process->txn_id(), $wip);

        return $outgoing;
    }

    public function enqueue_scripts($data) {
        
        $elavon_actions = $this->get_active_elavon_actions($data['form_id']);
        
        if (empty($elavon_actions)){
            
            return;
        }
     
        wp_enqueue_script('nf-elavon', NF_Elavon::$url . 'assets/js/elavon.js', array('nf-front-end'));
        
    }
    
    protected function get_active_elavon_actions($form_id) {
        
        $form_actions = Ninja_Forms()->form($form_id)->get_actions();
        
        $elavon_actions = array();
        
        foreach ($form_actions as $action) {

            if ('collectpayment' != $action->get_setting('type')) {
                continue;
            }

            if ($this->_slug != $action->get_setting('payment_gateways')) {
                continue;
            }

            $elavon_actions[] = array(
                'id' => $action->get_id(),
            );
        }
        return $elavon_actions;
    }

}
