<?php

/**
 * Static functions that format data for HTML output
 */
class NF_Elavon_Markup {

    /**
     * Adds markup and contextual help for given setting
     * 
     * @param type string
     * @param type mixed
     * @param type array
     * @return string
     */
    public static function markup($key, $raw,$contextual_help = null) {

        switch ($key) {

            case NF_Elavon_Constants::REQUEST_ARRAY:
                $markup = self::markup_request_array($raw);
                break;

            case NF_Elavon_Constants::RESPONSE_SUMMARY:
                $markup = self::markup_response_summary($raw,$contextual_help);
                break;

            case NF_Elavon_Constants::FULL_RESPONSE:
                $markup = self::markup_full_response($raw);
                break;

            default:
                $markup = '<h3>Markup Placeholder</h3>';
        }

        return $markup;
    }

    /**
     * 
     * @param array $response_data_array Array of support data given as an array 
     * keyed on module with a summary string 
     * 
     * @return string HTML table markup of support data 
     */
    public static function markup_response_summary($response_data_array, $contextual_help = null) {

        $table = '<table><tbody>';

        if (!empty($response_data_array)) {

            foreach ($response_data_array as $module => $summary_array) {

                if (!is_array($summary_array)) {
                    continue;
                }

                $table .= '<tr>';

                $table .= '<td style="word-break: break-all; word-wrap: break-word; vertical-align:top;"><strong>' . $module . '</strong><br />';
                foreach ($summary_array as $summary) {

                    $summary_with_icons = self::support_tags_to_icons($summary);
                    $summary_with_help = self::add_contextual_help($summary_with_icons, $contextual_help);
                    $table .= $summary_with_help . '<br />';
                }
                $table .= '</td></tr>';
            }
        }
        $table .= '</tbody></table>';

        return $table;
    }

    public static function markup_full_response($full_response) {

        $table = '<table ><tbody>';

        if (!empty($full_response)) {

            foreach ($full_response as $module => $module_response) {
                $table .= '<tr>';

                $table .= '<td><strong>' . $module . '</strong><br />';
                $table .='<span style="word-break: break-all; word-wrap: break-word;">';


                $table .= serialize($module_response);

                $table .= '</span></td></tr>';
            }
        }
        $table .= '</tbody></table>';

        return $table;
    }

    /**
     * 
     * @param array $request_array The request array keyed on module with
     * serialized data
     * 
     * @return string HTML table markup of request array data 
     */
    public static function markup_request_array($request_array) {

        $table = '<table><tbody>';

        if (!empty($request_array)) {

            foreach ($request_array as $module => $summary) {

                $table .= '<tr>';

                $table .= '<td><strong>' . $module . '</strong><br />';
                
                $formatted_summary = self::build_formatted_summary($summary);
                
                $table .= '<span style="word-break: break-all; word-wrap: break-word;">' . $formatted_summary. '</span></tr>';
            }
        }
        $table .= '</tbody></table>';

        return $table;
    }
 
    /**
     * Try to display summary as key-field pairs if possible
     * 
     * Summary is built as indexed array of key value pairs, so grab the
     * first index for display
     * If structure not recognized, return serialized incoming
     * 
     * @param array $incoming
     * $return string
     */
    public static function build_formatted_summary($incoming) {

        $default = serialize($incoming);

        if (!is_array($incoming) || empty($incoming)) {

            return $default;
        }

        if (!is_array($incoming[0]) || empty($incoming[0])) {

            return $default;
        }

        $temp = '';

        foreach ($incoming[0] as $key => $value) {

            if (is_array($value)) {

                $str_value = serialize($value);
            } else {
                $str_value = $value;
            }

            $temp .= $key . " : " . $str_value . '<br />';
        }

        $outgoing = $temp;

        return $outgoing;
    }

    /**
     * Replaces support code tags with icons
     * 
     * @param string $incoming
     * @return string
     */
    protected static function support_tags_to_icons($incoming){
        
        $wip = $incoming;
        
        $wip1 = str_replace(NF_Elavon_Constants::ACTION_REQUIRED, '<span style = "color: orange;" class="dashicons dashicons-warning"></span>', $wip);
        
        $wip2 = str_replace(NF_Elavon_Constants::HELPFUL_HINT, '<span style = "color: blue;" class="dashicons dashicons-arrow-right-alt"></span>', $wip1);
        
        $wip3 = str_replace(NF_Elavon_Constants::SUCCESS, '<span style = "color: green;" class="dashicons dashicons-yes"></span>', $wip2);
        
        $outgoing = $wip3;
  
        return $outgoing;
    }
    
    /**
     * Searches string for contextual help strings and adds marked up help
     * @param string $incoming
     * @param array $contextual_help
     * @return string
     */
    protected static function add_contextual_help($incoming, $contextual_help) {

        if (is_null($contextual_help)) {
            
            return $incoming;
        }

        $contextual_help_array = $contextual_help->set_and_search_context($incoming);
                
        if(empty($contextual_help_array)){
            
            return $incoming;
        }
        
        $help_markup = self::contextual_help_markup($contextual_help_array);
        
        $outgoing = $incoming.'<br />'.$help_markup;
        
        return $outgoing;
    }

    /**
     * Given array of help instructions, adds markup and returns
     * @param array $incoming
     * @return string
     */
    protected static function contextual_help_markup($incoming) {

        $exploded = implode('<br />', $incoming);
        
        $dash_icon_style = ' color: orange; font-size: 2em;';
        $td_style = 'color: blue;';
        $td_style .= ' word-wrap: break-word;';
        
        $outgoing = sprintf('<table><tr><td><span style = "%s" class="dashicons dashicons-lightbulb"></span></td><td style = "%s">%s</td></tr></table>', $dash_icon_style, $td_style, $exploded);

        return $outgoing;
    }

}
