<?php

/**
 * Static functions for the payment gateway
 * 
 * Does not call any class methods outside of its own plugin
 */
class NF_Elavon_Functions {

    /**
     * Extracts the field map data for the given Fields to Extracts and builds
     * a nested array of the data
     * 
     * @param array $complete_field_map_data Entire field map data from Action
     * @param array $fields_to_extract Array of keys for fields to extract from repeater
     * 
     * @return array Array of field map data as a nested array
     */
    public static function extract_field_map_data($complete_field_map_data, $fields_to_extract) {

        $field_map_data = array(); // initialize

        foreach ($complete_field_map_data as $field_array) {

            $array = array();

            foreach ($fields_to_extract as $field) {

                $array[$field] = $field_array[$field];
            }
            $field_map_data[] = $array;
        }

        return $field_map_data;
    }

    /**
     * Formats an option array for the drop down menu in the option repeater
     * 
     * 
     * @param type $field_map_array
     * @return array Drop down version of array for the option repeater
     */
    public static function build_field_map_dropdown($field_map_array) {

        foreach ($field_map_array as $key => $label_map_array) {

            $field_map_dropdown[] = array(
                'label' => $label_map_array['label'],
                'value' => $key,
            );
        }

        return $field_map_dropdown;
    }

    /**
     * Build the entry type dropdown for the option repeater
     * 
     * Uses only the array keys from the list
     * @param $entry_type_array array Entry types for the field
     * @return array Drop down array for entry type
     */
    public static function build_entry_type_dropdown($entry_type_array) {

        $entry_type_dropdown = array(); // initialize

        $array_keys = array_keys($entry_type_array);

        foreach ($array_keys as $key) {

            $entry_type_dropdown[] = array(
                'label' => $key,
                'value' => $key,
            );
        }

        return $entry_type_dropdown;
    }

    /**
     * Extracts the advanced codes from the ninja_forms_settings option
     * 
     * @return array Advanced codes array
     */
    public static function extract_advanced_codes() {

        $settings_key = NF_Elavon_Constants::ADVANCED_CODES_KEY;

        $advanced_codes_array = array(); //initialize

        $nf_settings_array = get_option(' ninja_forms_settings');

        if (isset($nf_settings_array[$settings_key])) {

            $advanced_codes_setting = $nf_settings_array[$settings_key];

            $advanced_codes_array = array_map('trim', explode(',', $advanced_codes_setting));
        }

        return $advanced_codes_array;
    }

    /**
     * Applies all configured filters from the Advanced Command setting
     * 
     * @param type $single_filter Apply a specific filter on demand
     * @return none
     */
    public static function add_advanced_command_filters($single_filter = '') {

        if (defined('DISABLE_ELAVON_ADVANCED_COMMANDS')) {

            return;
        }

        $advanced_codes_array = self::extract_advanced_codes();

        if (0 === strlen($single_filter)) {

            $filters = include  NFELAVON_PLUGIN_DIR . 'includes/Config/AdvancedCommandFilters.php';
        } else {

            $full_config = include NFELAVON_PLUGIN_DIR . 'includes/Config/AdvancedCommandFilters.php';

            $filters[$single_filter] = $full_config[$single_filter];
        }

        foreach ($advanced_codes_array as $code) {

            if (array_key_exists($code, $filters)) {

                $filter = $filters[$code]['filter'];
                $callback = $filters[$code]['filter_callback'];

                add_filter($filter, 'NF_Elavon_Functions::' . $callback);
            }
        }
    }

    /**
     * Returns the process to use by checking if alternate processing is enabled 
     * and if the alternate process is activated
     * Accepted values: 
     * alternate_processing_enabled , form_processing , validation_object
     * @param string process  
     * 
     */
    public static function get_alternate_process_value($process = 'alternate_processing_enabled') {

        $return_array = array(
            'alternate_processing_enabled' => false,
            'form_processing' => NF_Elavon_Constants::PROCESS_CLASS,
            'validation_object' => NF_Elavon_Constants::VALIDATION_CLASS,
        );

        if (!array_key_exists($process,$return_array)) {

            $process = 'alternate_processing_enabled';
        }

        $advanced_commands = NF_Elavon_Functions::extract_advanced_codes();

        $alternate_processing_command = 'enable_alternate_processing';

        if (!in_array($alternate_processing_command, $advanced_commands)) {

            return $return_array[$process];
        }

        $return_array['alternate_processing_enabled'] = true;

        switch ($process) {

            case 'form_processing':

                if (class_exists(NF_Elavon_Constants::PROCESS_CLASS.'Alternate')) {

                    $return_array['form_processing'] = NF_Elavon_Constants::PROCESS_CLASS.'Alternate';
                }
                break;

            case 'validation_object':
                if (class_exists(NF_Elavon_Constants::VALIDATION_CLASS. 'Alternate')) {

                    $return_array['validation_object'] = NF_Elavon_Constants::VALIDATION_CLASS .'Alternate';
                }
                break;
        }

        return $return_array[$process];
    }

    /**
     * Strips HTML tags, usually from text areas
     * 
     * @param mixed $field_data
     */
    public static function remove_html_tags($field_data) {

        $stripped_response = $field_data; // initialize

        $keep_tags = apply_filters('nfelavon_keep_html_tags', FALSE);

        if (!$keep_tags) {
            $decoded = html_entity_decode($field_data);
            $stripped = wp_strip_all_tags($decoded);
            $stripped_response = esc_html($stripped);
        }

        return $stripped_response;
    }

    public static function return_true() {

        return true;
    }

    /**
     * Iterates the Action's $data array to extract data
     * 
     * While cycling through the data fields, extracts the key-values for
     * a single field and (optionally) passes that array to a method that
     * can handle the single field.
     * 
     * @param array $keys_to_extract Keys to extract from $data['fields][]
     * @param array $data Incoming $data from the Action
     * @param object $object Instance of method for single field processing
     * @param string $method Method to process single field during iteration
     * @return array Array of the extracted key-value pairs
     */
    public static function extract_data($keys_to_extract, $data, $object = null, $method='') {

        $extracted_data = array();

        if (!isset($data['fields']) || !is_array($data['fields'])) {
            return array();
        }

        foreach ($data['fields'] as $id => $field) {

            $single_field = array();

            $single_field['id'] = $id;

            foreach ($keys_to_extract as $key) {

                if (isset($field[$key])) {
                    $single_field[$key] = $field[$key];
                } else {
                    $single_field[$key] = false;
                }
            }

            if (method_exists($object, $method)) {
                
                $object->$method($single_field);
            }

            $extracted_data[] = $single_field;
        }

        return $extracted_data;
    }

   /**
     * Given field type and form $data, returns first ID matching type
     */
    public static function get_field_id_by_type($field_type, $data) {

        foreach ($data['fields'] as $field) {

            if ($field_type == $field['type']) {

                return $field['id'];
            }
        }

        return false;
    }
    
    /**
     * Force use of IPv4 when using Elavon's myvirtualmerchant.com requests
     * 
     * 
     * @since 3.1.1
     * @param type $ch
     * @param type $r
     * @param string $url
     * @return type
     */
    public static function forceIp4v($ch, $r, $url) {

        // If AdvancedCommand disables this method, stop immediately
        if (in_array('disable_forced_ipv4', self::extract_advanced_codes())){
            return;
        }
        
        // If the request is not to myvirtualmerchant.com, stop immediately
        if (false === strpos($url, 'myvirtualmerchant.com')){
            return;
        }
    
        // Force IPv4 
        if( defined('CURLOPT_IPRESOLVE') && defined('CURL_IPRESOLVE_V4')) {
            
            curl_setopt($ch, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);
        }
    }

}

add_action('init', 'NF_Elavon_Functions::add_advanced_command_filters');
/**
 * Add method to alter cURL within wp_api_url
 * 
 * Added 3.1.1
 */
add_action('http_api_curl','NF_Elavon_Functions::forceIp4v' , 10, 3);