<?php

/**
 * Extracts authorization parameters and provides them for API access
 * 
 * Helps with troubleshooting and support messages; may determine auth level
 * based on extracted auth parameters
 *
 * @author stuartlb3
 */
class ElavonAuth {

    
    /**
     * Merchant ID
     * @var string
     */
    protected $merchant_id;

    /**
     * User ID
     * @var string
     */
    protected $user_id;

    /**
     * PIN
     * @var string
     */
    protected $pin;

    /**
     * The authorization level determined by highest setting stored
     * @var string
     */
    protected $auth_level = '';

    /**
     * Status message describing current state of authorization
     * @var string
     */
    protected $status = '';

    /**
     * Request mode - Production or Demo - in plugin settings
     * @var string
     */
    protected $request_mode;
    
    public function __construct() {

        $this->get_stored_credentials();
        
        $this->determine_request_mode();
    }

    /**
     * Returns all stored required credentials from ninja_forms_settings 
     * 
     * Uses Ninja_Forms() methods and returns failure if NF class not loaded
     */
    protected function get_stored_credentials() {

        $this->status = __('Elavon credentials have not be retrieved', 'ninja-forms-elavon-payment-gateway');

        if (!class_exists('Ninja_Forms')) {

            $this->merchant_id = '';
            $this->user_id = '';
            $this->pin = '';

            $this->auth_level = NF_Elavon_Constants::CREDENTIALS_INCOMPLETE;

            $this->status = __('Ninja Forms is not present', 'ninja-forms-elavon-payment-gateway');
            return;
        }

        $this->merchant_id = Ninja_Forms()->get_setting(NF_Elavon_Constants::MERCHANT_ID);
        $this->user_id = Ninja_Forms()->get_setting(NF_Elavon_Constants::USER_ID);
        $this->pin = Ninja_Forms()->get_setting(NF_Elavon_Constants::PIN);

        if (0 == strlen($this->merchant_id) || 0 == strlen($this->user_id) || 0 == strlen($this->pin)) {

            $this->auth_level = NF_Elavon_Constants::CREDENTIALS_INCOMPLETE;
            $this->status = __('Elavon account credentials are missing or incomplete.', 'ninja-forms-elavon-payment-gateway');

            return;
        }

        $this->status = __('Elavon account credentials are set and ready to attempt communication.', 'ninja-forms-elavon-payment-gateway');
        $this->auth_level = NF_Elavon_Constants::CREDENTIALS_COMPLETE;
    }

    /**
     * Determine from plugin settings the Mode of communication
     * 
     * Can be Production or Demo mode; uses different URLs accordingly
     */
    protected function determine_request_mode(){
        
        $this->request_mode = Ninja_Forms()->get_setting( NF_Elavon_Constants::REQUEST_MODE );       
    }
    
    /**
     * Returns true auth level is enough to proceed with next steps
     * 
     * For Elavon, if credentials are present, that is enough to continue as
     * it will be validated or denied during request
     * 
     * @return boolean
     */
    public function is_authorized() {

        if (NF_Elavon_Constants::CREDENTIALS_INCOMPLETE != $this->auth_level) {

            return true;
        } else {

            return false;
        }
    }

    /**
     * Returns the status message of the Auth Object
     * @return string
     */
    public function status() {

        return $this->status;
    }

    /**
     * Returns the auth level determined by credentials extracted
     * @return string
     */
    public function auth_level() {

        return $this->auth_level;
    }

    /**
     * Returns the authorization parameters keyed per API specification
     * 
     * @return array
     */
    public function authorization_parameters() {

        $auth_param_array = array(
            'ssl_merchant_id' => $this->merchant_id,
            'ssl_user_id' => $this->user_id,
            'ssl_pin' => $this->pin,
        );

        return $auth_param_array;
    }

    /**
     * Returns the request mode (eg Production or Demo)
     * 
     * @return string
     */
    public function request_mode() {

        return $this->request_mode;
    }

}
