<?php

/**
 * Request Payment Class
 * 
 * Given a keyed array of request parameters, makes a request and returns
 * response from the payment gateway
 */
class ElavonRequestPayment {

    /** Production URL for requests */
    const PRODUCTION_URL = 'https://www.myvirtualmerchant.com/VirtualMerchant/processxml.do?';

    /** Demo URL for requests */
    const DEMO_URL = 'https://demo.myvirtualmerchant.com/VirtualMerchantDemo/processxml.do?';

    /**
     * Array of keyed parameters for making the payment request
     * @var array
     */
    protected $request_array;

    /**
     * Authorization object
     * @var object
     */
    protected $auth;

    /**
     * Query string build from request array
     * @var string
     */
    protected $request_query_string; // the query sting 

    /**
     * Base URL for requests
     * @var string
     */
    protected $base_url;
    protected $raw_response; // the raw response from elavon

    /**
     * Given a keyed array of request data, make the payment request
     * 
     * @param array $request_array Keyed array of request parameters
     */

    function __construct($request_array, $auth = null) {

        $this->request_array = $request_array;

        $this->auth = $auth;

        $this->construct_query_string();

        $this->set_base_url();

        $this->communicate_with_elavon();
    }

    /**
     * Assemble all parameters and build query string from them
     */
    protected function construct_query_string() {

        $auth_array = $this->auth->authorization_parameters();

        $full_array = array_merge($this->request_array, $auth_array);

        $full_array['ssl_show_form'] = 'false';
        $full_array['ssl_transaction_type'] = 'ccsale';
        $full_array['ssl_result_format'] = 'ASCII';

        
        $xml = new SimpleXMLElement("<txn></txn>");

        foreach ($full_array as $key => $value) {

            $xml->addChild($key, $value);
        }
        $dom = dom_import_simplexml($xml);
        $temp =  $dom->ownerDocument->saveXML($dom->ownerDocument->documentElement);
        $this->request_query_string = 'xmldata=' . $temp;

    }

    /**
     * Gets request mode from auth object and sets base URL accordingly
     */
    protected function set_base_url() {

        $request_mode = $this->auth->request_mode();

        if (NF_Elavon_Constants::DEMO_MODE == $request_mode) {

            $this->base_url = self::DEMO_URL;
        } else {

            $this->base_url = self::PRODUCTION_URL;
        }
    }

    /* Internal Methods */

    /**
     * Request payment thru API
     */
    protected function communicate_with_elavon() {

        $this->command_url = $this->base_url . $this->request_query_string;

        $params = array(
            'sslverify' => true,
            'timeout' => 15,
            'redirection' => 2
        );
        
        $this->raw_response = wp_remote_post($this->command_url, $params);
    }

    protected function check_for_communication_errors() {

        if (is_wp_error($this->raw_response)) { // wp_remote_post error
            foreach ($this->raw_response->errors as $code => $message_array) {

                $this->error_array[$code] = implode(' , ', $message_array);
            }

            return;
        }

        $this->build_response_body_array();

        if (isset($this->response_body_array['errorCode'])) { // elavon sent an error message back
            $this->error_array[$this->response_body_array['errorCode']] = $this->response_body_array['errorMessage'];
        }
    }

    protected function build_transaction_array() {

        if (isset($this->response_body_array['ssl_result'])) {

            foreach ($this->response_body_array as $key => $value) {

                if ('' != $value) {

                    $this->transaction_array[$key] = $value;
                }
            }
        }
    }

    protected function build_transaction_result() {

        if (isset($this->response_body_array['ssl_result'])) {

            $this->transaction_result = $this->response_body_array['ssl_result'];
        } else {

            $this->transaction_result = 'UNSET';
        }
    }

    protected function build_response_body_array() {

        $body_string = $this->raw_response['body'];

        $ascii_string = str_replace(array("\n", "\r", "\r\n"), '&', $body_string);
        parse_str($ascii_string, $array);

        $this->response_body_array = $array;
    }

    /* Sets and Gets */

    public function set_site_options($site_options_array) {

        if (empty($site_options_array) || !is_array($site_options_array)) {
            return;
        }
        $this->site_options = array_merge($this->site_options, $site_options_array);

        return;
    }

    public function set_form_and_field_settings($form_and_field_settings_array) {

        $this->form_and_field_settings = array_merge($this->form_and_field_settings, $form_and_field_settings_array);

        return;
    }

    /**
     * Returns the raw response from the RequestPayment
     * 
     * False if empty, WP_Error object, 
     * @return boolean|object|array
     */
    public function get_raw_response() {

        if (empty($this->raw_response)) {
            return false;
        } else {
            return $this->raw_response;
        }
    }

}
