<?php

/**
 * Given form submission, processes payment
 *
 */
class ElavonProcessForm {

    /**
     * Form action settings
     * @var array
     */
    protected $action_settings;

    /**
     * Form ID
     * @var integer
     */
    protected $form_id;

    /**
     * Form data passed in action as $data
     * @var array
     */
    protected $data;

    /**
     * Credit card fields with field map and validation
     * 
     * @var array
     */
    protected $creditcard_fields;

    /**
     * Extracted field map data from action_settings
     * @var array
     */
    protected $field_map_data;

    /**
     * Keyed array of form data to request payment from payment gateway
     * @var array
     */
    protected $request_array = array();

    /**
     * Processed response, keyed with default configured values
     * @var array
     */
    protected $processed_response;

    /**
     * Variable for storing HTML receipt data
     * 
     * @var string
     */
    protected $html_receipt_field_keys = array();

    /**
     * Variable for storing plain text receipt data
     * 
     * @var string
     */
    protected $plaintext_receipt_field_keys = array();

    /**
     * Variable for storing transaction id data
     * 
     * @var string
     */
    protected $txn_id_field_keys = array();

    
    /**
     * Communication authorization object
     * @var object
     */
    protected $auth;

    /**
     * Communication logging object
     * @var object
     */
    protected $commlog;

    /**
     * HTML Receipt
     * @var string
     */
    protected $html_receipt = '';

    /**
     * Plain text receipt
     * @var string
     */
    protected $plaintext_receipt = '';

    /**
     * Transaction ID string
     * @var string
     */
    protected $txn_id = '';

    /**
     * Lookup array to replacement characters disallowed in XML
     * @var array
     */
    protected $xml_replacement;
    
    public function __construct($action_settings, $form_id, $data) {

        $this->action_settings = $action_settings;

        $this->data = $data;

        $this->form_id = $form_id;

        $this->commlog = new ElavonCommLog;

        $this->auth = new ElavonAuth;
        
        $this->buildXMLReplacement();

        $this->commlog->log_entry(NF_Elavon_Constants::RESPONSE_SUMMARY, 'Authorization', $this->auth->status());

        if (!$this->auth->is_authorized()) {

            $this->commlog->update(NF_Elavon_Constants::SUPPORT_DATA);

            return;
        }

        $this->build_request_array();

        $this->make_and_process_request();
        
        if (TRUE == $this->processed_response['api_error'] ||
                TRUE!= $this->processed_response['approved']) {

            $this->trigger_processing_error();
            
            $this->commlog->update(NF_Elavon_Constants::SUPPORT_DATA);
            return;
        }
        
        $this->buildReceipts();

        $this->commlog->update(NF_Elavon_Constants::SUPPORT_DATA);
    }

    /**
     * Build request array from submitted credit card fields and option repeater
     */
    protected function build_request_array() {

        $this->extract_field_map_data();

        $this->set_total();
        
        $this->addFieldMapToRequest();

        $this->iterate_form_data();

        $this->log_request_array();
    }

    /**
     * RequestPayment and Handle Response
     */
    protected function make_and_process_request() {
        $payment_request = new ElavonRequestPayment($this->request_array, $this->auth);

        $raw_response = $payment_request->get_raw_response();

        $response_object = new ElavonHandleResponse($raw_response);

        $this->processed_response = $response_object->get_processed_response();

        $this->commlog->log_entry(NF_Elavon_Constants::RESPONSE_SUMMARY, 'RequestPayment', $this->processed_response['api_message']);
        $this->commlog->log_entry(NF_Elavon_Constants::FULL_RESPONSE, 'RequestPayment', $this->processed_response['response_body']);
    }

    /**
     * Build receipts and store in submissions if requested
     */
    protected function buildReceipts() {

        $receipt = new ElavonBuildReceipt($this->processed_response['response_body']);

        $this->html_receipt = $receipt->html();

        $this->plaintext_receipt = $receipt->plaintext();

        $this->txn_id = $receipt->txn_id();
        
        $this->modifyReceiptSubmissionData();
    }

    /**
     * Extracts the option repeater field map data
     */
    protected function extract_field_map_data() {

        $fields_to_extract = NF_Elavon::config('FieldsToExtract');

        $this->field_map_data = NF_Elavon_Functions::extract_field_map_data($this->action_settings[NF_Elavon_Constants::FIELD_MAP_REPEATER_KEY], $fields_to_extract);
    }

    /**
     * Iterates field_map_data and adds keyed values to request array
     */
    protected function addFieldMapToRequest() {

        if (empty($this->field_map_data)) {

            return;
        }

        $field_map_array = NF_Elavon()->get_field_map_array();

        foreach ($this->field_map_data as $field_map) {

            if (!isset($field_map_array[$field_map['field_map']]['map_instructions']) ||
                    !isset($field_map['form_field']) ||
                    empty($field_map['form_field'])) {

                continue;
            }
            

            $value = str_replace($this->xml_replacement[0],$this->xml_replacement[1], $field_map[ 'form_field' ]);          

            $this->request_array[$field_map_array[$field_map['field_map']]['map_instructions']] = $value;
        }
    }

    /**
     * Determine request total and add it to request array
     */
    protected function set_total() {

        //TODO: determine how to select total from either form value or field map
        $total = $this->action_settings['payment_total'];

        $this->request_array['ssl_amount'] = $total;
    }

    /**
     * Iterates through the action $data, calling processing method on each
     */
    protected function iterate_form_data() {

        $keys_to_extract = NF_Elavon::config('DataFieldKeys');

        $this->creditcard_fields = NF_Elavon::config('CreditCardFields');

        NF_Elavon_Functions::extract_data($keys_to_extract, $this->data, $this, 'process_data_field');
    }

    /**
     * Passes a single field extracted from the $data to methods for processing
     * 
     * Public method b/c it is called by Functions class
     * 
     * @param array $single_extracted_field
     */
    public function process_data_field($single_extracted_field) {

        $this->extract_validate_cc_data($single_extracted_field);
        $this->storeReceiptIDs($single_extracted_field);
    }

    /**
     * Validates credit card data and adds to request array
     * 
     * @param array $extracted_field
     */
    protected function extract_validate_cc_data($extracted_field) {

        if (!in_array($extracted_field['type'], array_keys($this->creditcard_fields))) {

            return;
        }

        $value_in = $extracted_field['value'];

        $validation_object_name = NF_Elavon_Constants::VALIDATION_CLASS;

        if ($this->creditcard_fields[$extracted_field['type']]) {

            $temp = $value_in;

            foreach ($this->creditcard_fields[$extracted_field['type']]['validation_functions'] as $function_call) {

                if (!method_exists($validation_object_name, $function_call)) {

                    continue;
                }

                $temp = call_user_func(array($validation_object_name, $function_call), $temp);
            }

            $validated = $temp;
        } else {

            $validated = $value_in;
        }

        $this->request_array[$this->creditcard_fields[$extracted_field['type']]['map_instructions']] = $validated;
    }

    /**
     * If extracted field is set as the html field, store its ID and key
     * 
     * Field key is set as advanced action setting in pre-3.0; after 3.0,
     * merge tags are used to insert the receipts and transaction ID into 
     * email or success messages.
     * 
     * To store the value in the submission, the merge tag value must be set
     * as the field LABEL (usually a hidden field since it isn't completed
     * by the form user).  Label is used because the format of a merge tag
     * can't be used on a field key.
     * 
     * 
     * @param array $single_extracted_field
     */
    protected function storeReceiptIDs($single_extracted_field) {

        if ($single_extracted_field['key'] == $this->action_settings[NF_Elavon_Constants::HTML_RECEIPT_FIELD_KEY] || // pre-3.0
                $single_extracted_field['label'] == NF_Elavon_Constants::HTML_RECEIPT_FIELD_KEY ) {

            $this->html_receipt_field_keys = array(
                'id' => $single_extracted_field['id'],
                'key' => $single_extracted_field['key'],
            );
        }
        
        if ($single_extracted_field['key'] == $this->action_settings[NF_Elavon_Constants::PLAINTEXT_RECEIPT_FIELD_KEY] || // pre-3.0
                $single_extracted_field['label'] == NF_Elavon_Constants::PLAINTEXT_RECEIPT_FIELD_KEY) {

            $this->plaintext_receipt_field_keys = array(
                'id' => $single_extracted_field['id'],
                'key' => $single_extracted_field['key'],
            );
        }

        if ($single_extracted_field['key'] == $this->action_settings[NF_Elavon_Constants::TXN_ID_FIELD_KEY] || // pre-3.0
              $single_extracted_field['label'] == NF_Elavon_Constants::TXN_ID_FIELD_KEY  ) {

            $this->txn_id_field_keys = array(
                'id' => $single_extracted_field['id'],
                'key' => $single_extracted_field['key'],
            );
        }
    }

    /**
     * Mask the credit card details and log entry
     */
    protected function log_request_array() {

        $masked_request = $this->request_array;

        $keys_to_mask = array(
            'creditcardnumber',
            'creditcardcvc',
            'creditcardexpiration',
            'creditcardzip',
        );

        foreach ($keys_to_mask as $key) {
            $field_to_mask = $this->creditcard_fields[$key]['map_instructions'];
            if (isset($masked_request[$field_to_mask])) {

                $masked_request[$field_to_mask] = '***';
            }
        }
        $this->commlog->log_entry(NF_Elavon_Constants::REQUEST_ARRAY, 'CC_sale', $masked_request);
    }

    /**
     * Overwrites the submission value for the receipt keys and txn id
     */
    protected function modifyReceiptSubmissionData() {

        if (isset($this->html_receipt_field_keys['id'])) {

            $this->updateSubmission(array(
                $this->html_receipt_field_keys['id'] => $this->html_receipt
            ));
        }

        if (isset($this->plaintext_receipt_field_keys['id'])) {

            $this->updateSubmission(array(
                $this->plaintext_receipt_field_keys['id'] => $this->plaintext_receipt
            ));
        }

        if (isset($this->txn_id_field_keys['id'])) {

            $this->updateSubmission(array(
                $this->txn_id_field_keys['id'] => $this->txn_id
            ));
        }
    }

    /**
     * Updates the submission with field-keyed values
     * 
     * @param array $data_to_update Key is field ID, value is new value
     */
    protected function updateSubmission($data_to_update = array()) {

        if (!isset($this->data['actions']['save']['sub_id'])) {
            return;
        }

        $sub_id = $this->data['actions']['save']['sub_id'];
        $sub = Ninja_Forms()->form()->sub($sub_id)->get();

        foreach ($data_to_update as $key => $value) {
            $sub->update_field_value($key, $value)->save();
        }
        $sub->save();
    }

    protected function trigger_processing_error(){
        
        $msg = $this->processed_response['error_message']; // set default
        $field_id = false;
        
        switch ($this->processed_response['error_code']) {

            case '5001': // Exp Date Invalid
   
                $field_id = NF_Elavon_Functions::get_field_id_by_type('creditcardexpiration', $this->data);
                break;

            case '5021': // Invalid CVV2 Value
                $field_id = NF_Elavon_Functions::get_field_id_by_type('creditcardcvc', $this->data);
                break;
            
            case '5000': // Credit Card Number Invalid
            case '9999': // Only test cards allowed
            case 'DECL': // Response came back without APPROVAL value 
                $field_id = NF_Elavon_Functions::get_field_id_by_type('creditcardnumber', $this->data);
                break;

            default:

                break;
        }

        if(!$field_id){
            $this->data[ 'errors' ][ 'form' ][ 'elavon' ] = $msg;
            return;
        }
        
        $this->data[ 'errors' ][ 'fields' ][ $field_id ] = array( 'message' => $msg, 'slug' =>  NF_Elavon_Constants::ACTION_KEY);
        return;
    }
    
    /**
     * Builds array of values to replace and what replaces them
     * 
     * @since 3.1.0
     */
    protected function buildXMLReplacement()
    {
        $this->xml_replacement = apply_filters('nfelavon_xml_replacement', array(
            array( '#'),
            array( '' ),
        ));
    }

    /**
     * Returns the action data
     * @return array
     */
    public function data() {

        return $this->data;
    }

    /**
     * Return the HTMl receipt
     * @return string
     */
    public function html_receipt() {

        return $this->html_receipt;
    }

    /**
     * Return the plain text receipt
     * @return string
     */
    public function plaintext_receipt() {

        return $this->plaintext_receipt;
    }

   /**
     * Return the transaction id
     * @return string
     */
    public function txn_id() {

        return $this->txn_id;
    }
}
