<?php

/**
 * Handles the response from the API
 *
 */
class ElavonHandleResponse {

    /**
     * Complete response from API or WP (if wp_error)
     * @var mixed
     */
    protected $raw_response;

    /**
     * Response after processing
     * @var array
     */
    protected $processed_response;

    /**
     * The response code from the API
     * @var integer
     */
    protected $response_code;

    /**
     * The response message from the API
     * @var integer
     */
    protected $response_message;

    /**
     * Extracted headers from the API
     * @var array
     */
    protected $headers = array();

    /**
     * Response body XML
     * @var object
     */
    protected $body;

    /**
     * Keys of transaction response
     * @var array
     */
    protected $txn_fields;

    /**
     * Array of response key field pairs from payment gateway
     * @var array
     */
    protected $txn_array;
    
    
    public function __construct($raw_response) {

        $this->raw_response = $raw_response;

        // set default
        $this->processed_response = NF_Elavon::config('ResponseArray');

        $this->wp_error_check();

        if ($this->processed_response['wp_error']) {
            return;
        }

        $this->extract_headers();

        $this->response_code = $this->raw_response['response']['code'];

        $this->response_message = $this->raw_response['response']['message'];

        $this->extract_body_xml();

        $this->api_error_check();

        if ($this->processed_response['api_error']) {
            return;
        }

        $this->txn_fields = NF_Elavon()->config_once('TransactionResponse');

        $this->extract_txn_array();
        
        $this->set_approval();
        
        $this->build_api_message();
    }

    /**
     * Builds keyed array of header responses
     */
    protected function extract_headers() {

        if (isset($this->raw_response['headers']['date'])) {

            $this->headers['date'] = $this->raw_response['headers']['date'];

            $this->processed_response['timestamp'] = $this->headers['date'];
        }

        $this->processed_response['headers'] = $this->headers;
    }

    /**
     * Extracts XML body and convert into keyed array
     */
    protected function extract_body_xml() {

        $incoming_use_errors = libxml_use_internal_errors(true);

        $xml = new SimpleXMLElement($this->raw_response['body']);

        $json = json_encode($xml);

        $this->body = json_decode($json, TRUE);

        libxml_use_internal_errors($incoming_use_errors);

        $this->processed_response['response_body'] = $this->body;
    }

    /**
     * Extract data from XML response into a transaction array
     * 
     * Cycles thru oonfig'd txn_fields to set default of missing values
     * 
     */
    protected function extract_txn_array() {

        foreach ($this->txn_fields as $key=>$config){
            
            $this->txn_array[$key] = $config;
            
            //overwrite default value with API response value if present
            // implode if value is array
            if(isset($this->processed_response['response_body'][$key])){
                               
                if(is_array($this->processed_response['response_body'][$key])){
                    
                    $temp = implode(' ',$this->processed_response['response_body'][$key]);
                }else{
                    
                    $temp = $this->processed_response['response_body'][$key];
                }
                
                $this->txn_array[$key]['value']=$temp;
            }
        }
    }    
    
    /**
     * Build API message based on returned values
     */
    protected function build_api_message(){
        if (0 < strlen($this->txn_array['ssl_result_message']['value'])) {

            $msg = '';
            $msg .= $this->txn_array['ssl_result_message']['value'] . ': ';
            $msg .= $this->txn_array['ssl_card_number']['value'] . ' - ';
            $msg.= $this->txn_array['ssl_first_name']['value'] . ' ' . $this->txn_array['ssl_last_name']['value'] . ' - ';
            $msg.= $this->txn_array['ssl_txn_time']['value'];
 
            $this->processed_response['api_message'] = $msg;
            
            return;
        }

        // foreign conversion
        if (0 < strlen($this->txn_array['ssl_conversion_rate']['value'])) {

            $msg = '';
            $msg .= $this->txn_array['ssl_amount']['value'] . __(' rate by  ', 'ninja-forms-elavon-payment-gateway');
            $msg .= $this->txn_array['ssl_dcc_rate_provider']['value'] . ':';

            $msg .= __(' Markup: ', 'ninja-forms-elavon-payment-gateway') . $this->txn_array['ssl_markup']['value'];
            $msg .= __(' Conversion: ', 'ninja-forms-elavon-payment-gateway') . $this->txn_array['ssl_markup']['value'];
            $msg .= __(' Cardholder amount: ', 'ninja-forms-elavon-payment-gateway') . $this->txn_array['ssl_cardholder_amount']['value'];

            $this->processed_response['api_message'] = $msg;
           
            return;
        }

        $this->processed_response['api_message'] = maybe_serialize($this->txn_array);
        return;
    }

    /**
     * Checks for API error
     */
    protected function api_error_check() {

        if (!isset($this->body['errorCode'])) {

            return;
        }

        $this->processed_response['error'] = true;
        $this->processed_response['api_error'] = true;
        $this->processed_response['error_code'] = $this->body['errorCode'];

        $this->determine_error_message();
    }

    /**
     * Checks for WordPress error
     */
    protected function wp_error_check() {

        if (is_wp_error($this->raw_response)) {

            $this->processed_response['error'] = true;
            $this->processed_response['wp_error'] = true;
            $this->processed_response['error_code'] = 'wp';
            $this->processed_response['error_message'] = $this->raw_response->get_error_message();
            $this->processed_response['api_message'] = NF_Elavon_Constants::ACTION_REQUIRED . $this->raw_response->get_error_message();
        }
    }

    /**
     * If error, see if response body has more descriptive message for support
     */
    protected function determine_error_message() {

        $msg = '';

        if (isset($this->body['errorName'])) {

            $msg .= $this->body['errorName'] . ' - ';
        }

        if (isset($this->body['errorMessage'])) {

            $msg .= $this->body['errorMessage'];
        }

        $msg = rtrim($msg, ' - ');

        $this->processed_response['error_message'] = $msg;
        $this->processed_response['api_message'] = NF_Elavon_Constants::ACTION_REQUIRED . $msg;
    }

    /**
     * Checks for APPROVAL and sets response values
     */
    protected function set_approval() {

        if ( 'APPROVAL' == $this->txn_array['ssl_result_message']['value']) {

            $this->processed_response['approved'] = true;
            return;
        }
        
        $this->processed_response['error'] = true;
        $this->processed_response['error_code'] = 'DECL';
        $this->processed_response['error_message'] = __('Card was not approved for sale','ninja-forms-elavon-payment-gateway');
    }

    /**
     * The complete processed response array
     * @return array
     */
    public function get_processed_response() {

        return $this->processed_response;
    }

}
