<?php

/**
 * Builds receipts from API response
 *
 * @author stuartlb3
 */
class ElavonBuildReceipt {

    /**
     * Array of response key field pairs from payment gateway
     * @var array
     */
    protected $txn_array;

    /**
     * Configured transaction fields with settings
     * @var array
     */
    protected $txn_fields = array();

    /**
     * Indexed array of key-value pairs of receipt information
     * @var array
     */
    protected $receipt_array;

    /**
     * HTML receipt markup
     * @var string
     */
    protected $html_receipt = '';

    /**
     * Plain text receipt
     * @var string
     */
    protected $plaintext_receipt = '';

    /**
     * Transaction ID string
     * @var string
     */
    protected $txn_id = '';
    
    /**
     * Given transcaction response, builds html and plain text receipts
     * 
     * @param array $txn_array Transaction response from Payment Gateway
     */
    function __construct($txn_array) {

        $this->txn_array = $txn_array;

        $this->txn_fields = apply_filters('nfelavon_filter_receipt_order', NF_Elavon()->config_once('TransactionResponse'));

        $this->build_receipt_fields();

        $this->build_receipts();
    }

    /**
     * Build indexed array of receipt fields in order to be displayed
     * 
     * If txn_array isn't set for a given key, skip over it
     * if receipt order is set to 999, skip over it
     */
    protected function build_receipt_fields() {

        foreach ($this->txn_fields as $key => $field_data_array) {

            if (!isset($this->txn_array[$key]) || 999 == $field_data_array['receipt_order']) {
                continue;
            }

            $this->receipt_array[$field_data_array['receipt_order']] = array(
                'key' => $key,
                'value' => $this->txn_array[$key],
            );
        }
    }

    /**
     * Iterates receipt array and builds html and plain text receipts
     */
    protected function build_receipts() {

        if (empty($this->receipt_array)) {
            return;
        }

        $this->html_receipt .= '<div class="receipt-wrapper">';

        foreach ($this->receipt_array as $receipt_field) {

            $stripped_key = $this->strip_key_prefix($receipt_field['key']);

            $this->html_receipt .= '<span class = "label-' . $stripped_key . '">' . $stripped_key . ': </span>';
            $this->html_receipt .= '<span class = "value-' . $stripped_key . '">' . $receipt_field['value'] . '</span><br />';

            $this->plaintext_receipt .= $stripped_key . ': ' . $receipt_field['value'] . "\r\n";
            
            if('txn_id'==$stripped_key){
                
                $this->txn_id = $receipt_field['value'];
            }
        }

        $this->html_receipt .= '</div>';
    }

    /**
     * Strips prefix from field keys for readability
     * 
     * @param string $key
     * @return string The key stripped of Payment Gateway's prefix
     */
    protected function strip_key_prefix($key) {

        $stripped_key = str_replace('ssl_', '', $key);

        return $stripped_key;
    }

    /**
     * Returns the HTML receipt
     * 
     * @return string
     */
    public function html() {

        return $this->html_receipt;
    }

    /**
     * Returns the plain text receipt
     * 
     * @return string
     */
    public function plaintext() {

        return $this->plaintext_receipt;
    }

    /**
     * Returns the transaction ID
     * 
     * @return string
     */
    public function txn_id() {

        return $this->txn_id;
    }
}
